/* Copyright (c) 2016-2018 VMware, Inc. All rights reserved. */
package com.vmware.automatic.plugin.registration;

import com.vmware.automatic.plugin.registration.commands.IsPluginRegisteredCmd;
import com.vmware.automatic.plugin.registration.commands.PluginCmd;
import com.vmware.automatic.plugin.registration.commands.RegisterPluginCmd;
import com.vmware.automatic.plugin.registration.commands.UnregisterPluginCmd;
import com.vmware.automatic.plugin.registration.commands.UpdatePluginCmd;
import com.vmware.automatic.plugin.registration.commands.utils.PrintHelper;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * Main class for invoking plugin registration commands.
 */
public class PluginRegistrationMain {
   private static final Map<String, PluginCmd> commands = new HashMap<String, PluginCmd>();
   private static final Options options;
   private static String _allCommands;

   /**
    * Add commands and options
    */
   static {
      commands.put("registerPlugin", new RegisterPluginCmd());
      commands.put("unregisterPlugin", new UnregisterPluginCmd());
      commands.put("updatePlugin", new UpdatePluginCmd());
      commands.put("isPluginRegistered", new IsPluginRegisteredCmd());
      options = new Options();
      Iterator<String> keyIterator = commands.keySet().iterator();
      while (keyIterator.hasNext()) {
         Options commandOptions = commands.get(keyIterator.next()).getPluginOpts();
         for (Option o : commandOptions.getOptions()) {
            o.setRequired(false);
            options.addOption(o);
         }
      }

      _allCommands = Arrays.toString(commands.keySet().toArray());
      _allCommands = _allCommands.substring(1, _allCommands.length() - 1);
      options.addOption(Option.builder("action").hasArg().argName("action")
            .desc(_allCommands).required().build());
   }

   public static void main(String[] args) {
      execute(args);
   }

   /**
    * Parse arguments, find command class and execute.
    *
    * @param args command line arguments
    */
   private static void execute(String[] args) {
      CommandLine commandLine = null;
      CommandLineParser parser = new DefaultParser();
      try {
         commandLine = parser.parse(options, args, true);
      } catch (ParseException e) {
         PrintHelper.printHelp(options, e.getMessage());
      }
      if (commandLine != null) {
         PluginCmd pluginCmd = commands.get(commandLine.getOptionValue("action"));
         if (pluginCmd != null) {
            try {
               pluginCmd.execute(args);
            } catch (Exception e) {
               e.printStackTrace();
            }
         } else {
            PrintHelper.printHelp(options, "Missing argument for option: -action <action> " + _allCommands);
         }
      }
   }
}
